'use client'

import { useState, useEffect } from 'react'
import { Button } from '@/components/ui/button'
import { Card, CardContent } from '@/components/ui/card'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import { 
  MapPin, 
  Phone, 
  Mail, 
  Calendar, 
  Users, 
  Car,
  Shield,
  Clock,
  Star,
  ChevronRight,
  Menu,
  X
} from 'lucide-react'
import Link from 'next/link'
import ChatSupport from '@/components/ChatSupport'

export default function Home() {
  const [isMenuOpen, setIsMenuOpen] = useState(false)
  const [cars, setCars] = useState([])
  const [settings, setSettings] = useState(null)

  useEffect(() => {
    fetchCars()
    fetchSettings()
  }, [])

  const fetchCars = async () => {
    try {
      const response = await fetch('/api/cars?featured=true&limit=6')
      const data = await response.json()
      if (data.success) {
        setCars(data.data)
      }
    } catch (error) {
      console.error('Failed to fetch cars:', error)
    }
  }

  const fetchSettings = async () => {
    try {
      const response = await fetch('/api/settings')
      const data = await response.json()
      if (data.success) {
        setSettings(data.data)
      }
    } catch (error) {
      console.error('Failed to fetch settings:', error)
    }
  }

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-0 w-full bg-white/95 backdrop-blur-sm z-50 border-b border-gray-100">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Link href="/" className="flex items-center space-x-2">
                <Car className="h-8 w-8 text-blue-600" />
                <span className="text-xl font-bold text-gray-900">
                  {settings?.companyName || 'Rental Mobil'}
                </span>
              </Link>
            </div>

            <div className="hidden md:flex items-center space-x-8">
              <Link href="#home" className="text-gray-700 hover:text-blue-600 transition">Beranda</Link>
              <Link href="/cars" className="text-gray-700 hover:text-blue-600 transition">Armada</Link>
              <Link href="#about" className="text-gray-700 hover:text-blue-600 transition">Tentang</Link>
              <Link href="#contact" className="text-gray-700 hover:text-blue-600 transition">Kontak</Link>
              <Link href="/admin">
                <Button variant="outline">Admin</Button>
              </Link>
            </div>

            <button
              className="md:hidden"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
            >
              {isMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
            </button>
          </div>
        </div>

        {/* Mobile menu */}
        {isMenuOpen && (
          <div className="md:hidden bg-white border-t border-gray-100">
            <div className="px-2 pt-2 pb-3 space-y-1">
              <Link href="#home" className="block px-3 py-2 text-gray-700 hover:text-blue-600">Beranda</Link>
              <Link href="/cars" className="block px-3 py-2 text-gray-700 hover:text-blue-600">Armada</Link>
              <Link href="#about" className="block px-3 py-2 text-gray-700 hover:text-blue-600">Tentang</Link>
              <Link href="#contact" className="block px-3 py-2 text-gray-700 hover:text-blue-600">Kontak</Link>
              <Link href="/admin" className="block px-3 py-2">
                <Button variant="outline" className="w-full">Admin</Button>
              </Link>
            </div>
          </div>
        )}
      </nav>

      {/* Hero Section */}
      <section id="home" className="pt-16 min-h-screen flex items-center relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-blue-50 to-white"></div>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <Badge className="mb-4 bg-blue-100 text-blue-800">
                🚗 Rental Mobil Terpercaya
              </Badge>
              <h1 className="text-4xl lg:text-6xl font-bold text-gray-900 mb-6">
                Solusi Rental Mobil 
                <span className="text-blue-600"> Profesional</span>
              </h1>
              <p className="text-xl text-gray-600 mb-8">
                Layanan rental mobil terbaik dengan armada terlengkap, 
                harga terjangkau, dan pelayanan prima untuk kebutuhan perjalanan Anda.
              </p>
              <div className="flex flex-col sm:flex-row gap-4">
                <Link href="/cars">
                  <Button size="lg" className="bg-blue-600 hover:bg-blue-700">
                    <Calendar className="mr-2 h-5 w-5" />
                    Lihat Armada
                  </Button>
                </Link>
                <Button size="lg" variant="outline">
                  <Phone className="mr-2 h-5 w-5" />
                  Hubungi Kami
                </Button>
              </div>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-blue-100 to-blue-50 rounded-2xl p-8">
                <Car className="h-64 w-64 text-blue-600 mx-auto" />
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features */}
      <section className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-4">
              Mengapa Memilih Kami?
            </h2>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto">
              Kami memberikan layanan terbaik dengan berbagai keunggulan untuk kenyamanan Anda
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            <Card className="text-center p-6 border-0 shadow-sm">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Shield className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Terpercaya</h3>
                <p className="text-gray-600">Layanan aman dan terjamin dengan reputasi terbaik</p>
              </CardContent>
            </Card>
            <Card className="text-center p-6 border-0 shadow-sm">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Clock className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">24/7 Service</h3>
                <p className="text-gray-600">Layanan tersedia setiap saat saat Anda butuhkan</p>
              </CardContent>
            </Card>
            <Card className="text-center p-6 border-0 shadow-sm">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Users className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Profesional</h3>
                <p className="text-gray-600">Tim berpengalaman yang siap melayani Anda</p>
              </CardContent>
            </Card>
            <Card className="text-center p-6 border-0 shadow-sm">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Star className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Berkualitas</h3>
                <p className="text-gray-600">Armada terbaru dengan kondisi prima</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Featured Cars */}
      <section id="cars" className="py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-4">
              Armada Unggulan Kami
            </h2>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto">
              Pilihan mobil terbaik untuk berbagai kebutuhan perjalanan Anda
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {cars.map((car) => (
              <Card key={car.id} className="overflow-hidden hover:shadow-lg transition">
                <div className="aspect-video bg-gray-100 relative">
                  {car.image ? (
                    <img 
                      src={car.image} 
                      alt={car.name}
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    <div className="w-full h-full flex items-center justify-center">
                      <Car className="h-16 w-16 text-gray-400" />
                    </div>
                  )}
                  <Badge className="absolute top-4 right-4 bg-blue-600">
                    Rp {car.price.toLocaleString('id-ID')}/hari
                  </Badge>
                </div>
                <CardContent className="p-6">
                  <h3 className="text-xl font-semibold mb-2">{car.name}</h3>
                  <p className="text-gray-600 mb-4">{car.brand} {car.model}</p>
                  <div className="flex items-center justify-between text-sm text-gray-500 mb-4">
                    <span>{car.year}</span>
                    <span>{car.seats} Seats</span>
                    <span>{car.transmission}</span>
                  </div>
                  <Link href={`/cars/${car.id}`}>
                    <Button className="w-full">
                      Lihat Detail
                      <ChevronRight className="ml-2 h-4 w-4" />
                    </Button>
                  </Link>
                </CardContent>
              </Card>
            ))}
          </div>
          <div className="text-center mt-12">
              <Link href="/cars">
                <Button size="lg" variant="outline">
                  Lihat Semua Armada
                  <ChevronRight className="ml-2 h-5 w-5" />
                </Button>
              </Link>
          </div>
        </div>
      </section>

      {/* About */}
      <section id="about" className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-6">
                Tentang Perusahaan Kami
              </h2>
              <p className="text-lg text-gray-600 mb-6">
                {settings?.about || 
                  'Kami adalah perusahaan rental mobil terkemuka yang telah berpengalaman selama bertahun-tahun dalam menyediakan layanan transportasi terbaik untuk berbagai kebutuhan Anda.'}
              </p>
              <p className="text-lg text-gray-600 mb-8">
                Dengan komitmen untuk memberikan pelayanan prima dan armada berkualitas, 
                kami menjadi pilihan utama untuk kebutuhan rental mobil di Indonesia.
              </p>
              <div className="grid grid-cols-2 gap-6">
                <div>
                  <h3 className="text-3xl font-bold text-blue-600 mb-2">500+</h3>
                  <p className="text-gray-600">Armada Mobil</p>
                </div>
                <div>
                  <h3 className="text-3xl font-bold text-blue-600 mb-2">10,000+</h3>
                  <p className="text-gray-600">Pelanggan Puas</p>
                </div>
              </div>
            </div>
            <div className="bg-gradient-to-br from-blue-100 to-blue-50 rounded-2xl p-8">
              <div className="grid grid-cols-2 gap-4">
                <div className="bg-white rounded-xl p-6 text-center">
                  <Shield className="h-12 w-12 text-blue-600 mx-auto mb-2" />
                  <h4 className="font-semibold">Asuransi</h4>
                </div>
                <div className="bg-white rounded-xl p-6 text-center">
                  <Clock className="h-12 w-12 text-blue-600 mx-auto mb-2" />
                  <h4 className="font-semibold">Support 24/7</h4>
                </div>
                <div className="bg-white rounded-xl p-6 text-center">
                  <Car className="h-12 w-12 text-blue-600 mx-auto mb-2" />
                  <h4 className="font-semibold">Mobil Baru</h4>
                </div>
                <div className="bg-white rounded-xl p-6 text-center">
                  <Users className="h-12 w-12 text-blue-600 mx-auto mb-2" />
                  <h4 className="font-semibold">Driver Profesional</h4>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Contact */}
      <section id="contact" className="py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl lg:text-4xl font-bold text-gray-900 mb-4">
              Hubungi Kami
            </h2>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto">
              Siap melayani kebutuhan rental mobil Anda
            </p>
          </div>
          <div className="grid lg:grid-cols-3 gap-8">
            <Card className="text-center p-6">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Phone className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Telepon</h3>
                <p className="text-gray-600">{settings?.phone || '+62 812-3456-7890'}</p>
              </CardContent>
            </Card>
            <Card className="text-center p-6">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Mail className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Email</h3>
                <p className="text-gray-600">{settings?.email || 'info@rentalmobil.com'}</p>
              </CardContent>
            </Card>
            <Card className="text-center p-6">
              <CardContent className="pt-6">
                <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <MapPin className="h-8 w-8 text-blue-600" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Alamat</h3>
                <p className="text-gray-600">{settings?.address || 'Jakarta, Indonesia'}</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid md:grid-cols-4 gap-8">
            <div>
              <div className="flex items-center space-x-2 mb-4">
                <Car className="h-8 w-8 text-blue-400" />
                <span className="text-xl font-bold">
                  {settings?.companyName || 'Rental Mobil'}
                </span>
              </div>
              <p className="text-gray-400">
                Solusi rental mobil terpercaya untuk kebutuhan perjalanan Anda.
              </p>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Layanan</h4>
              <ul className="space-y-2 text-gray-400">
                <li>Rental Harian</li>
                <li>Rental Mingguan</li>
                <li>Rental Bulanan</li>
                <li>Antar Jemput</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Perusahaan</h4>
              <ul className="space-y-2 text-gray-400">
                <li>Tentang Kami</li>
                <li>Armada</li>
                <li>Promo</li>
                <li>Karir</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Ikuti Kami</h4>
              <div className="flex space-x-4">
                {settings?.facebook && (
                  <a href={settings.facebook} className="text-gray-400 hover:text-white">
                    Facebook
                  </a>
                )}
                {settings?.instagram && (
                  <a href={settings.instagram} className="text-gray-400 hover:text-white">
                    Instagram
                  </a>
                )}
                {settings?.twitter && (
                  <a href={settings.twitter} className="text-gray-400 hover:text-white">
                    Twitter
                  </a>
                )}
              </div>
            </div>
          </div>
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2024 {settings?.companyName || 'Rental Mobil'}. All rights reserved.</p>
          </div>
        </div>
      </footer>

      {/* Chat Support */}
      <ChatSupport />
    </div>
  )
}